
(* preuve intuitioniste d'après Roy Dyckhoff
   
   https://philpapers.org/s/Roy%20Dyckhoff
   https://www.jstor.org/stable/2275431?seq=1
   https://www.cs.cmu.edu/~fp/courses/atp/cmuonly/D92.pdf
   et localement dans le dossier Roy_Dyckhoff

   *)
open Interface;; 


(* fonctions générales sur les listes *)
let aplatir g =
  (* g a list list
     résultat concaténation des listes de g *)
  List.fold_left (@) [] g;;

let espaces k =
  (* k : int
     résultat : k espaces *)
  if k < 0 then " " else String.make k ' ';;

(*  redefini dans le module preuve
    let assoc1 x a =
  (* a : 'a * int list est une liste de couples, l'élement droite de chaque couple est strictement positif
     résultat :
     i si le couple (x,i) est dans la liste 
     0 sinon
  *)
  try List.assoc x a
  with Not_found -> 0;;

 *)

let rm_last l =
(* l : 'a list
   résultat l sans son dernier élément
   *)
  List.rev (try List.tl (List.rev l) with Failure _ -> []);;

 
let ind = 2;;
(* valeur de l'indentation quand on ajoute une hypothèse 
     utilisée dans la construction et l'annotation des preuves
     Assistant.verifier_ligne, Preuve.afficher_ligne 

*)

let mg = 6;;
(* colonne du début des preuves (après le numéro de ligne) 
     utilisée seulement dans l'annotation pour laisser
     la place des numéros de ligne
*)	

let col_justification = 60;;
(* colonne des justifications 
     utilisée dans la construction et l'annotation des preuves
     Assistant.verifier_ligne, Preuve.afficher_ligne 
*)

let ecart = 10 ;;
(* borne de l'écart entre la fin d'une formule et la colonne des justifications 
     utilisée dans la construction et l'annotation des preuves
     Assistant.verifier_ligne, Preuve.afficher_ligne 
*)


let col_hyp = 80;;
(* colonne de début de l'affichage des listes d'hypothèses
   non utilisée actuellement
*)



let rec entiers_du_contexte  g =
  (* g : formula*int list
     résultat la suite des entiers extraits de g dans l'ordre inverse *)
  match g with
    | [] -> ""
    |[(_,n)] -> string_of_int n
    |(_,n)::fg -> entiers_du_contexte fg^","^string_of_int n ;;


	
(* Affichage des listes de lignes *)

let afficher_ligne tab ligne = 
(* Effet : la ligne est affichée avec la tabulation !tab
   tab est modifiée par Assume (augmentée de ind), Therefore (diminuée de ind)
*)
match ligne with
  | Assume a -> 
      (let (_ , sf)= string_of_formula (!tab) (col_justification -ecart) (!tab+10) 0 a in
	print_string ((espaces !tab)^"assume "^sf^".\n"));
      tab :=!tab+ind
  | Therefore a ->
      tab :=!tab-ind;
      let (_ , sf)= string_of_formula (!tab) (col_justification -ecart) (!tab+10) 0 a in
	print_string ((espaces !tab)^"therefore "^sf^".\n")
  | End a ->
      tab :=!tab-ind;
      let (_ , sf)= string_of_formula (!tab) (col_justification -ecart) (!tab+10) 0 a in
	print_string ((espaces !tab)^"end "^sf^".\n")
  | Usable a ->
      let (_ , sf)= string_of_formula (!tab) (col_justification -ecart) (!tab) 0 a in
	print_string ((espaces !tab)^sf^".\n")
;;


let afficher_preuve p =
(* p : line list 
   Effet : affichage de la preuve p en colonne 1 
   pour afficher en colonne k initialiser tab à la valeur k-1    
*)
  let tab = ref 0 in
    List.iter (afficher_ligne tab) p;;


(* les lemmes *)

let p1 b c = 
  (* preuve de Neg b sachant Neg (Disj (b,c)) *)
  [Assume b; Usable (Disj (b,c)); Usable False ; Therefore (Neg b)]
;;

  
let p2 b c =
  (* preuve de Neg c sachant Neg (Disj (b,c)) *)
  [Assume c; Usable (Disj (b,c)); Usable False ; Therefore (Neg c)]
;;



let p3 b c =
  (* preuve de Disj(Neg b,c) sachant Imp(b,c) *)
  [Assume (Neg (Disj(Neg b,c)))]@(p1 (Neg b) c)@[Usable b]@(p2 (Neg b) c)@
    [Usable c; Usable False; Therefore (Neg (Neg (Disj(Neg b,c))));Usable (Disj(Neg b,c))]
;;			

let p5 b c =
  (* preuve de Disj (Neg b, Neg c) sachant Neg (Conj (b,c)) *)
  [ Assume (Neg (Disj (Neg b,Neg c)));
    Assume (Neg b); Usable (Disj (Neg b, Neg c)); Usable False ; Therefore (Neg (Neg b));
    Usable b;
    Assume (Neg c); Usable (Disj (Neg b, Neg c)); Usable False ; Therefore (Neg (Neg c));
    Usable c;
    Usable (Conj (b,c)); Usable False ;
    Therefore (Neg (Neg (Disj (Neg b, Neg c)))); Usable (Disj (Neg b, Neg c))]
;;

let p6 b c =
  (* preuve de b sachant Neg (Imp (b,c)) *)
  [ Assume (Neg b); 
    Assume b ; Usable False; Usable c; Therefore (Imp (b,c)); Usable False ; 
    Therefore (Neg (Neg b)) ; Usable b]
;;



let p7 b c =
  (* preuve de Neg c sachant Neg (Imp (b,c)) *)
  [ Assume c;
    Assume b; Therefore (Imp (b,c)); Usable False ; 
    Therefore  (Neg c)]
;;

let p8 b c =
  (* preuve de Imp (Neg b,c) sachant Neg (Equiv (b,c)) *)
  [ Assume (Neg b); 
    Assume (Neg c); 
    Assume b ; Usable False; Usable c; Therefore (Imp (b,c));
    Assume c ; Usable False; Usable b; Therefore (Imp (c,b)); Usable (Equiv (b,c)); Usable False ;
    Therefore (Neg (Neg c)); Usable c; Therefore (Imp (Neg b,c))]
;;

let p9 b c =
  (* preuve de Imp (c, Neg b) sachant Neg (Equiv (b,c)) *)
  [ Assume c;
    Assume b; 
    Assume b; Therefore (Imp (b,c)); Assume c; Therefore (Imp (c,b));
    Usable (Equiv (b,c)); Usable False ; 
    Therefore (Neg b);  Therefore (Imp (c, Neg b))]
;;





let p10 b c d =
  (* preuve de Imp (c,Imp(d,b)) sachant Imp(Conj(c,d),b) *)
  [ Assume c; Assume d; Usable (Conj (c,d)); Usable b; 
    Therefore (Imp (d,b)); Therefore (Imp (c, Imp(d,b)))]
;;

let p11 b c d =
  (* preuve de Imp (c,b) sachant Imp (Disj (c,d),b) *)
  [ Assume c; Usable (Disj (c,d)); Usable b; Therefore (Imp(c,b))]
;;

let p12 b c d =
  (* preuve de Imp (d,b) sachant Imp (Disj (c,d),b) *)
  [ Assume d; Usable (Disj (c,d)); Usable b; Therefore (Imp(d,b))]
;;

let p13 b c d =
  (* preuve de Imp (d,b) sachant Imp (Imp (c,d),b) *)
  [ Assume d; Assume c; Therefore (Imp (c,d)); Usable b; Therefore (Imp (d,b))]
;;


let p14 b c =
   (* preuve de -(b=>c)+-(c=>b) depuis -(b <=> c) *)
   [Assume (Neg (Disj (Neg (Imp (b,c)), Neg (Imp(c,b))))); Assume (Neg(Imp(b,c))); Usable (Disj (Neg(Imp(b,c)),(Neg(Imp(c,b)))));
    Usable False; Therefore (Neg (Neg (Imp(b,c)))); Usable (Imp(b,c));
    Assume (Imp(c,b)); Usable (Equiv (b,c));Usable False; Therefore (Neg (Imp(c,b))); Usable (Disj(Neg (Imp(b,c)),Neg (Imp(c,b))));
    Usable False; Therefore (Neg (Neg (Disj (Neg(Imp(b,c)),Neg(Imp(c,b)))))); Usable (Disj(Neg(Imp(b,c)),Neg(Imp(c,b))))]



exception Improvable of formula list ;;

(* fonctions auxiliaires pour les raisonnements en avant *)
let intersection l1 l2 =
  (* l1, l2 : 'a list
     resultat : intersection des deux listes *)
  List.fold_left (fun l el2 -> if List.mem el2 l1 then el2::l else l) [] l2;;

let remove x l =
  (* l : 'a list
     resultat : la liste obtenue en enlevant de l les éléments égaux à x *)
   List.fold_left (fun l el2 -> if el2 <> x then el2::l else l) [] l;;

let implications gamma a =
  (* gamma : formula list
     a : formula
     resultat : liste des formules x telles que Imp (x,a) est dans gamma *)
  List.fold_left (fun l el2 -> match el2 with | Imp (x, y) when y = a -> x::l |_ -> l) [] gamma;;

let modusponens gamma a =
  (* gamma : formula list
     a : formula
     resultat : true ssi il y a une formule b de gamma telle que b et Imp (b,a) dans gamma *)
  intersection gamma (implications gamma a)  <> [];;

let negations gamma gamma' =
  (* gamma : formula list   
     resultat : liste des formules x telles que x est dans gamma et Neg x est dans gamma' *)
  List.fold_left (fun l el2 -> if List.mem (Neg el2) gamma' then el2::l else l ) [] gamma;;

let contradictions gamma =
  (* gamma : formula list
     a : formula
     resultat : true ssi il y a une formule b de gamma telle que b et Neg b dans gamma *)
  negations gamma gamma <> [];;

let conjonctions gamma a =
  (* gamma : formula list
     a : formula
     resultat : true ssi il y a une formule de gamma de la forme Conj (_,a) ou Conj (a,_) *)
  List.fold_left (fun c el2 -> match el2 with | Conj (x,y) when x = a || y = a -> true | _ -> c) false gamma;;

(* Modification de preuve : le 17 juillet 2008 
On utilise les abréviations *)

let rec deplie a = 
  (* a formule 
     résultat a sans negation ni equivalence *)
  match a with
    | Var _ | False -> a
    | Imp (b,c) -> Imp(deplie b, deplie c)
    | Disj (b,c) -> Disj(deplie b, deplie c)
    | Conj (b,c) -> Conj(deplie b, deplie c)
    | Neg b -> Imp(deplie b, False)
    | Equiv (b,c) -> let db = deplie b and dc = deplie c in Conj(Imp(db,dc),Imp(dc,db));;

let deplie_liste la =
  (* la liste de formules
     resultat la liste des formules deplies *)
  List.map deplie la;;





let negs gamma =
  (* gamma : formula list 
     resultat : liste des Neg x tels que Imp (x,_) dans gamma *)
  List.fold_left (fun l el2 -> match el2 with | Imp (x,_) -> (Neg x):: l |_ -> l) [] gamma;;

let implicationsgauches gamma a =
  (* gamma : formula list
     a : formula
     resultat : liste des formules x telles que Imp(a,x) est dans lf *)
  List.fold_left (fun l el2 -> match el2 with | Imp (y,x) when y = a -> x::l | _ -> l) [] gamma;;


  
(* Construction d'une preuve intuitioniste suivant la méthode de Roy *)


let rec preuve non_examinees atomes implications_atomiques implications_doubles a =
  (* non_examinees formula list
     atomes formula list ne comportant que des atomes
     implications_atomiques formula list de la forme Imp(Var _, _)
     implications_doubles formula list de la forme Imp(Imp(_,_),_)
     a formula
     résultat 
     posons gamma = non_examinees @ atomes @ implications_atomiques @ implications_doubles
     - une preuve intuitioniste (line list) de a sachant gamma
     exception lorsque la preuve n'existe pas 
     - Improvable ll où ll est une  liste de littéraux 
            modèle de gamma  et contre-modèle de a
   *)
  
  
  let da = deplie a and dgamma =  deplie_liste (non_examinees @ atomes @ implications_atomiques @ implications_doubles) in
  if (* raisonnement en avant en zero ou un pas : attention une preuve peut être vide *)
    List.mem da dgamma then []
  else if
    List.mem False dgamma || modusponens dgamma da || conjonctions dgamma da then [Usable a]
  else if modusponens dgamma False then [Usable False; Usable a]
  else 
  (* raisonnement en arrière              *)

    match non_examinees, a with
      (* Traitement des conclusions, sauf Disj _  non réversible et  False, Var _  *)
	
      | _, Imp (b,c) ->
	  let r = preuve (b::non_examinees) atomes implications_atomiques implications_doubles c in
	     [Assume b] @ r @ [Therefore a]

      | _, Neg b -> 
	  let r = preuve (b::non_examinees) atomes implications_atomiques implications_doubles False in
	  [Assume b] @ r @ [Therefore a]
	  
      | _, Conj (b,c) ->
	  let r = preuve non_examinees atomes implications_atomiques implications_doubles b   and
	    s = preuve non_examinees atomes implications_atomiques implications_doubles c in
	    r @ s @ [Usable a]

      | _, Equiv (b,c) ->
	  let r = preuve non_examinees atomes implications_atomiques implications_doubles (Imp (b,c)) and
	    s = preuve non_examinees atomes implications_atomiques implications_doubles (Imp (c,b)) in
	    r @ s @ [Usable a]

		      (* Traitement des hypothèses *)

      | ((Var _) as y)::non_examinees',_ ->
	  preuve non_examinees' (y::atomes) implications_atomiques implications_doubles a
	    
      | (Conj (b,c)) :: non_examinees', _ ->
	  let r = preuve (b::c::non_examinees') atomes implications_atomiques implications_doubles a in
	  [Usable b;Usable c] @ r

      | (Disj (b,c)) :: non_examinees', _ ->
	  (* let r = preuve (b::non_examinees') atomes implications_atomiques implications_doubles a and
	      s = preuve (c::non_examinees') atomes implications_atomiques implications_doubles a in
              [Assume b] @ r @ [Therefore (Imp (b,a)); Assume c] @ s @ [Therefore (Imp (c,a)); Usable a] *)

              let r =  (preuve (b::non_examinees') atomes implications_atomiques implications_doubles a) and
	      s =  (preuve (c::non_examinees') atomes implications_atomiques implications_doubles a) in

              [Assume b] @ r @ [End a; Assume c] @ s @ [End a; Usable a]
     
      | (Equiv (b,c)) :: non_examinees', _ ->
	  (* On remplace (b <=> c) par (b => c) /\ (c => b) *)
          preuve ((Conj (Imp(b,c),Imp(c,b)))::non_examinees') atomes implications_atomiques 
	    implications_doubles a 
						       
      | (Neg b) :: non_examinees', _ ->
	  (* Version ancienne : On remplace  Neg b par Imp (b,False) 
             Version nouvelle : de -b on deduit b => F
           *)
	 preuve ((Imp (b,False)) ::non_examinees') atomes implications_atomiques implications_doubles a
                
 
      | ((Imp (g,b))as y)::non_examinees', _ ->
          (* On tente une preuve directe de Imp(g,b) en prouvant g puis une preuve de a sachant b
	     et en cas d'échec, on utilise les règles de LJT de Roy Dickhoff
	     Le cas particulier est celui de la règle =>G 1 où g est élément de non_examinés'
	     et où pg = []
	   *)
         begin
	   try
	     let pg = preuve non_examinees' atomes implications_atomiques implications_doubles g
	     in let pa = preuve (b::non_examinees') atomes implications_atomiques implications_doubles a
	        in pg @ [Usable b] @ pa
	   with Improvable _ -> 

	    match g with
	      
	      | (Var x)  ->
		preuve non_examinees' atomes (y::implications_atomiques) implications_doubles a

	      | Conj (c,d) ->
		(* LJT p797 =>G 2 *)
		    let paux = preuve (Imp (c, Imp(d,b))::non_examinees')
		      atomes implications_atomiques implications_doubles a 
		    in (p10 b c d) @ paux

	      | Disj (c,d) ->
		(* LJT p797 =>G 3 *)
		    let paux = preuve (Imp (c,b)::Imp (d,b)::non_examinees')
		      atomes implications_atomiques implications_doubles a 
		    in (p11 b c d)@ (p12 b c d) @ paux

	      | Imp (c,d) -> 
		    preuve non_examinees' atomes implications_atomiques (y::implications_doubles) a

	      | Neg c ->
		  preuve non_examinees' atomes implications_atomiques  (Imp (Imp(c,False),b)::implications_doubles) a 
                 
	      | Equiv (c,d) -> (* remplacement de c <=> d par (c => d) /\ (d => c) *)
                 preuve (Imp (Conj (Imp(c,d),Imp(d,c)),b)::non_examinees') atomes implications_atomiques implications_doubles a

	      | False  ->
		    (* False => b peut-être enlevée *)
		    preuve non_examinees' atomes implications_atomiques implications_doubles a

           end
      
      | [], a ->
	  (* toutes les hypothèses ont été examinées ainsi que toutes les conclusions sauf
	     celles de la forme False, Var _, Disj _  
	     On applique la règle L0=> réversible

	   *)
   
	 let rec l0imp variables = match variables with
	   | [] -> non_reversibles atomes implications_atomiques implications_doubles a
	   | var::fvar -> 
	      let lg = implicationsgauches implications_atomiques var in
	      (* lg est la liste des b tels que  Imp (var,b) est dans la liste implications_atomiques *)
	      if lg = [] then l0imp fvar
	      else let b = List.hd lg in
	      let paux = preuve [b] atomes (remove (Imp (var,b)) implications_atomiques) 
		implications_doubles a
	      in Usable b :: paux
	 in l0imp atomes
      | _ -> failwith "erreur dans preuve"

  and
    non_reversibles atomes implications_atomiques implications_doubles a =
  (* préconditions : 
     Aucune déduction n'est possible sans l'emploi des règles non_réversibles, en particulier
     la règle L0=> n'est pas applicable, autrement dit il n'y a pas de x tel que Var x dans
     la liste atomes et Imp(Var x,_) dans la liste implications_atomiques 
     la conclusion  a est de l'un des formes False, Var _, Disj _ *)
  let rec limpimp ids = match ids with
  | [] -> (match a with
    | False -> raise (Improvable (atomes @ negs implications_atomiques))

    | Var _ -> (* a n'est pas dans gamma donc pas dans atomes *)
		raise (Improvable ((Neg a)::atomes @ negs implications_atomiques))

    |  Disj (b,c) -> 
	(try let pb = preuve [] atomes implications_atomiques implications_doubles b in 
	pb @ [Usable a] 
	with
	| Improvable _ -> 
	    let pc = preuve [] atomes implications_atomiques implications_doubles c in
	    pc @ [Usable a] ))
  | ((Imp (Imp(c,d),b))as y)::fids ->
      let reste_implications_doubles = remove y implications_doubles in
      try 
	let paux1 = preuve [c; Imp(d,b)] atomes implications_atomiques reste_implications_doubles d
	and 
	    paux2 = preuve [b] atomes implications_atomiques reste_implications_doubles a
	in
	(p13 b c d) @ [Assume c] @ paux1 @ [Therefore (Imp (c, d)); Usable b] @ paux2
      with Improvable _ -> limpimp fids

  in limpimp implications_doubles

;;

	    


(* Construction d'une preuve en logique classique suivant la méthode décrite
   dans le livre Logique et démonstration automatique paragraphe tactiques de preuve p77 *)

let rec preuvec hnv  hl a =
(* l'environnement pour prouver a est divisée en deux listes
   hnv est la liste des hypothèses non vues encore à décomposer
   hl est une liste des hypothèses qui sont des littéraux *)
let da = deplie a and dgamma = deplie_liste hnv@hl in
if List.mem da dgamma then []
(* Remarque : une preuve de a dans l'environnement gamma est une preuve qui montre que gamma |- a.
Elle peut être vide et a n'est donc pas nécécessairement le dernier élément de la preuve *)
else if (* a déduite avec une seule règle appliquée à gamma *)
List.mem False dgamma (* a déduite par la règle Efq *)
|| modusponens dgamma da (* Il y a une formule b et (b => a) dans gamma, d'où a déduite par la règle =>E *)
|| conjonctions dgamma da (* Il y a une formule b telle que (b & a) ou (a & b) dans gamma, d'où a déduite par la règle &E1 ou &E2 *)
|| contradictions dgamma (* Il y a une formule b telle que b et -b sont éléments de gamma, d'où a déduite par la règle -E,  a,-a - donc b  *)
then [Usable a]
else if 
modusponens dgamma False (* Il y a une formule b telle que b et (b => F) dans gamma, d'où F est déduite de gamma, puis a par la règle Efq *)
then [Usable False; Usable a]
else (* On décompose a où a est l'une des formules b=>c, -b, b&c, b<=>c *)
match a with
   | Imp(b,c) ->
     (* gamma !- b=> c si et seulement si gamma,b !- c
     Soit r une preuve de gamma,b !- c. Alors une preuve de ce que gamma !- b=> c est [Assume b]@r@[Therefore (b=>c)] *)
     let r = preuvec (b::hnv)  hl c in [Assume b]@r@[Therefore a]
   | Neg b ->
     let r = preuvec (b::hnv) hl False in [Assume b]@r@[Therefore a]
   | Conj(b,c) ->
     (* gamma !- b&c si et seulement si gamma !- b et gamma !-b
        Soit r une preuve de b depuis gamma et s une preuve de c depuis gamma. Alors une preuve de ce que gamma !- b&c est
        r@s@[a] *)
     let r = preuvec hnv  hl b and s = preuvec hnv  hl c in r@s@[Usable a]
   | Equiv(b,c) -> 
     let r = preuvec hnv  hl (Imp(b,c)) and s = preuvec hnv  hl (Imp(c,b)) in r@s@[Usable a]
   | Disj (b,c) -> (* On tente de prouver b puis c et en cas d'échec, on exécute la fonction preuve_hnv qui décompose les hypothèses *)
      (try let pb = preuvec hnv  hl b in pb@[Usable a]
      with
      | Improvable _ ->
        try let pc = preuvec hnv  hl c in pc@[Usable a]
        with
        | Improvable _ -> (* décomposition des hypothèses par appel de preuve_hnv *) preuve_hnv hnv  hl a)
  | _ -> (* a est F ou une variable donc non décomposable *) preuve_hnv hnv  hl a


and (* La fonction preuve_hnv recherche une preuve de a en décomposant les hypothèses de hnv. Lorsque la liste hnv est vide, on décompose le
       but a dans les seuls cas restant *)
    preuve_hnv hnv  hl a =
if hnv = [] then
  match a with
  | False -> raise (Improvable hl)
  | Var _ -> raise (Improvable ((Neg a)::hl))
  | Disj (b,c) ->
    (* On décompose a via l'équivalence gamma !- b+c si et seulement si gamma, -b !- c.
       Soit r une preuve de c dans l'environnement gamma, -b. Une preuve de b+c depuis gamma est
       [Assume -a]@ preuve de -b dans l'environnement -a@ r @ [a;F;Therefore --a;a ] *)
       let r = preuvec [Neg b]  hl c in
       [Assume (Neg a); Assume b; Usable a; Usable False; Therefore (Neg b)]@r@[Usable a;Usable False;Therefore (Neg (Neg a));Usable a]
else (* Décomposition des hypothèses *)
let hyp1::fin_hnv = hnv in
match hyp1 with (* On décompose les hypothèses b&c, b<=>c, b+c, b => c ,-b (avec plusieurs cas suivant b) *)
| False -> [Usable a]
| Var _ -> preuve_hnv fin_hnv  (hyp1::hl) a
| Neg (Var _) -> preuve_hnv fin_hnv  (hyp1::hl) a
| Conj(b,c) -> 
   (* gamma, b&c !- a si et seulement si gamma,b,c !- a.
      Soit r une preuve de a dans l'environnement gamma,b,c. Alors une preuve de a dans l'envoronnement gamma, b&c est
      [b;c]@r *)
    let r = preuvec (b::c::fin_hnv)  hl a in [Usable b;Usable c]@r
| Equiv (b,c) ->
    let r = preuvec (Imp(b,c)::Imp(c,b)::fin_hnv)  hl a in [Usable (Imp (b,c));Usable (Imp (c,b))]@r
| Disj (b,c) -> 
   (* gamma,b+c !- a si et seulement si gamma,b !-a et gamma,c !- a 
      Soit r une preuve de a depuis gamma,b et s une preuve de a depuis gamma, c . Alors une preuve de a depuis gamma,b+c est
      [Assume b]@r@[End a;Assume c]@s@[End a;Usable a]. Notons que le dernier a sera déduit par la règle +E de b+c, et des conditions b!-a et c!-a *)
   let r = preuvec (b::fin_hnv)  hl a and s = preuvec (c::fin_hnv)  hl a in 
      [Assume b]@r@[End a;Assume c]@s@[End a;Usable a]
| Imp (b,c) -> 
    (try
      let pb = preuvec fin_hnv hl b in
      let pa = preuvec (c::fin_hnv) hl a in pb@[Usable c]@pa
     with Improvable _ ->
      (* gamma,b=>c !- a si et seulement si gamma,-b !-a et gamma,c !- a
      Soit r une preuve de a depuis gamma,-b et s une preuve de a depuis gamma,c. Alors un preuve de a depuis gamma,b=>c est
      (preuve (p3 b c) de -b+c depuis  b=> c)@[Assume -b]@r@[End a; Assume c@s@[End a;Usable a]*)
       let r = preuvec ((Neg b)::fin_hnv)  hl a and s = preuvec (c::fin_hnv)  hl a in 
       (p3 b c)@[Assume (Neg b)]@r@[End a;Assume c]@s@[End a;Usable a])

(* Différence avec la version DN1 : dans DN1 on évite (p3 b c) qui a une preuve longue *)

      (* Restent à décomposer les hypothèses -b où b n'est pas une variable *)
| Neg False -> (* Hypothèse inutile *) preuve_hnv fin_hnv  hl a
| Neg (Neg b) ->
       (* gamma,--b !- a si et seulement si gamma,b !- a.
         Soit r une preuve de a depuis gamma,b alors [Usable b]@r est une preuve de a depuis gamma, --b *)
       let r = preuvec (b::fin_hnv)  hl a in [Usable b]@r
| Neg (Imp(b,c)) ->
      (* gamma,-(b=>c)!- a si et seulement si gamma,b,-c !- a.
         Soit r une preuve de a depuis gamma,b,-c. Soit (p6 b c) une preuve de b depuis -(b=>c), (p7 b c) une preuve de -c depuis -(b=>c).
	 Alors (p6 b c)@(p7 b c)@r est une preuve de a depuis gamma,-(b=>c) *)
      let r = preuvec (b::(Neg c)::fin_hnv)  hl a in    (p6 b c)@(p7 b c)@r
| Neg (Conj(b,c)) ->
      (* gamma,-(b&c) !- a si et seulement si gamma,-b+-c|-a.
         Soit r une preuve de a depuis gamma,-b+-c. Soit (p5 b c) une preuve de -b+-c depuis -(b&c).
         Alors (p5 b c)@r est une preuve de a depuis gamma,-(b&c) *)
      let r = preuvec (Disj(Neg b,Neg c)::fin_hnv)  hl a in (p5 b c)@r
| Neg (Equiv(b,c)) -> (* Une équivalence est une conjonction *)
      (* gamma,-(b<=>c) !-a si et seulement si gamma, -(b=>c)+-(c=>b)!- a
         Soit r une preuve de a depuis gamma,-(b=>c)+-(c=>b).
         Alors (p14 b c) une preuve de -(b=>c)+-(c=>b) depuis -(b <=> c) alors (p14 b c)@r est une preuve de a depuis gamma,-(b<=>c) !- a *)
       let r = preuvec (Disj(Neg(Imp(b,c)), Neg(Imp(c,b)))::fin_hnv)  hl a in (p14 b c)@r
| Neg (Disj(b,c)) ->
      (* gamma,-(b+c) !-a si et seulement si gamma,-b,-c !- a
         Soit r une preuve de a depuis gamma,-b,-c. Alors
	 (p1 b c)@(p2 b c)@r est une preuve de a depuis -(b+c),gamma, car (p1 b c) déduit -b de -(b+c) et (p2 b c) déduit -c de -(b+c) *)
      let r = preuvec (Neg b::Neg c::fin_hnv)  hl a in (p1 b c)@(p2 b c)@r
       
;;


let preuve_classique a =
   (* a est une formule
      resultat : si a est classiquement prouvable alors une preuve de a
      exception : Improvable ll une liste de littéraux contre-modèle de a 
      si a est prouvable intuitionistiquement, on en donne une preuve sans raa
      sinon on recherche une preuve classique, si possible  courte (ce qui est
      subjectif) sans utiliser le théorème de Glivenko

   *)
   try preuve [] [] [] [] a
    with | Improvable _ ->
            ( preuvec []  [] a)
   ;;

   




let rec compacter l =
  (* suppression des éléments répétés de l*)
  match l with
    | [] -> []
    | x::m -> if List.mem x m then compacter m else x::compacter m;;

let rec afficher_liste_litteraux l =
  (* l formula list
     effet : affichage des éléments de l séparés par des virgules 
     exception : levée par string_of_litteral si une des formules de l
       n'est pas un littéral
  *)
  match l
  with
    | [] -> ()
    | [a] ->print_string (string_of_litteral a); 
    | a::fl ->print_string (string_of_litteral a);print_string ",";afficher_liste_litteraux fl
;;  
     

	  





   
   






	    
	 
    
    


