open List;;
open Tableau;;

(* on définit des priorités dans l'ordre d'applications des règles
   - 1 : les règles qui ne créent pas de cas à analyser
   - 2 : les règles qui créent des cas à analyser
   - 3 : la règle du possible, qui peut créer un nouvel état
*)

exception Error_priority;;

let priority = function 
  | (Pos _) -> 3
  (* disjonctions *)
  | (Or _) -> 2 
  | (Imp _) -> 2
  | (Equ _) -> 2
  | (Not (And _)) -> 2
  | (Not (Equ _)) -> 2
  (* non disjonctions *)
  | (Nec _) -> 1
  | (And _) -> 1
  | (Not False) ->0
  | (Not (Or _)) -> 1
  | (Not (Imp _)) -> 1
  | (Not (Nec _)) -> 1
  | (Not (Pos _)) -> 1
  | (Not (Not _)) -> 1
  | (Var _) -> 1
  | (Not (Var _)) -> 1
  | _ -> raise Error_priority;;

(* On distingue les hypothèses de type int*formula et les hypothèses numérotées 
de type int*(int*formula) dont le premier entier sert à numéroter les hypothèses
pour l'affichage des preuves *)

(* la priorité des formules est étendue à celles des hypothèses numérotées d'une branche *)

(* le nouvel infeg_assumption est nécessité par le remplacement de Sort.list par List.sort 
let infeg_assumption = fun (_,(_,f1)) (_,(_,f2)) -> (priority f1 <= priority f2) ;;
*)

let infeg_assumption = fun (_,(_,f1)) (_,(_,f2)) -> compare (priority f1) (priority f2) ;;

(* Soit n un entier et assumptions une liste d'hypothèses 
   dont les états sont au plus égaux à n 
   (rearrange n assumptions) reorganise cette liste en un tableau dont la composante i est 
   l'ensemble des formules d'états i *)

let rearrange n assumptions =
  let t = Array.make (n+1) [] in
    (iter (function (p,a) -> t.(p)<-a::t.(p)) assumptions; t);;

let rearrangen n numeroted_assumptions =
  let t = Array.make (n+1) [] in
    (iter (function (n,(p,a)) -> t.(p)<-(n,a)::t.(p)) numeroted_assumptions; t);;




(* (included l1 l2) si la liste l1 est incluse dans la liste l2 *)

let included l1 l2 = 
  let rec included_aux = function
    |[] -> true 
    | a::l1 -> (mem a l2) && included_aux l1
  in included_aux l1;;


(* (existe e t) 
   lève l'exception (Index i) si la liste e est incluse dans t.(i)
   sinon donne faux  *)

exception Index of int;;

let existe e t =
  let n = Array.length t in
    for i=0 to n-1 do
      if included e t.(i) then raise (Index i)
    done;
    false;;



(* element : 'a -> int*'a list -> bool*int
   element x lan = (true,k) si lan comporte (k,x)
                         = (false,0) sinon
*)

let element x lan = 
  let rec aux = function
    | [] -> (false,0)
    | (n,y)::u -> if x = y  then (true, n) else aux u
  in aux lan;;

(* contradiction : int*(int*formula) list -> bool *(int*(int*int))
   contradiction lan = (false,(0,(0,0))) si lan ne comporte pas de contradiction
                     = (true,(e,(k,l)))si les hypothèses de numéros k et l et d'état e
                        sont contradictoires
*)

let rec contradiction = function
  | (n,(e,Not a))::u -> let (b,q) = element (e,a) u in if b then (true,(e,(n,q))) else contradiction u
  | (n,(e,a))::u -> let (b,q) = element (e,Not a) u in if b then (true,(e,(n,q))) else contradiction u
  | [] -> (false,(0,(0,0)));;

(* falseonbranch : int*(int*formule) list -> bool
   falseonbranch lan = false si lan ne comporte pas la formule F 
                     = true  si lan comporte F
*)

let rec falseonbranch = function
  | (n,(e,False))::u -> true
  | _ :: u -> falseonbranch u
  | [] -> false;;


let rec s4_of_int a =
  (* s4_of_int a est la traduction en S4 de la formule intuitioniste a 
     meilleure est la traduction, meilleurs seront les modèles construits
     à partir de cette traduction.
     On donne un exemple de cette amélioration ci-dessous
  *)
  match a with
    | False -> False
    | Var _ -> Nec a
    | Or (b,c) -> 
	(* Or (s4_of_int b,s4_of_int c)
	   On tente de diminuer le nombre de modalités
	   par l'équivalence -[]p + -[]q = -[](p.q) *)	
	let b' = s4_of_int b and c' = s4_of_int c in
	  (match (b',c') with
	      Not (Nec b''),Not (Nec c'') -> Not (Nec (And (b'',c'')))
	    | _ -> Or(b',c'))		
    | And (b,c) -> 
	(* And(s4_of_int b,s4_of_int c)
	   On tente de diminuer le nombre de modalités
	   par l'équivalence []p + []q = [](p.q) *)

	let b' = s4_of_int b and c' = s4_of_int c in
	  (match (b',c') with
	      Nec b'',Nec c'' -> Nec (And (b'',c''))
	    | _ -> And(b',c'))
    | Equ (b,c) -> s4_of_int (And (Imp (b,c),Imp(c,b)))
	(* Nec (Equ (s4_of_int b,s4_of_int c)) 
	   car Nec et And commutent en logique modale *)
    | Imp (b,c) -> 
	(* Nec (Imp (s4_of_int b,s4_of_int c)) *)
	(* dans S4 []([]p => []q) équivaut à []([]p => q) *)
	let c' = s4_of_int c in
	(match c' with
	  | Nec c'' -> Nec (Imp (s4_of_int b,c''))
	  | _ -> Nec (Imp (s4_of_int b, c')))
    | Not b -> Nec (Not (s4_of_int b))
    | _ -> failwith "erreur dans s4_of_int" ;; 





(* (complete b) si la branche b est sans contradiction et 
si toutes ses formules sont marquées *)

let complete b = (b.todo = []) && 
  let (b,r) = (contradiction b.is_done)in not b  ;;

(* put_off_first_component [(a1,b1);...(an,bn)] = ([b1;...bn] *)
let put_off_first_component l = rev (fold_left (fun l (_,n) -> n :: l) [] l);;

exception Error_satisfy;;

(* On extrait de la liste l de formules numerotées celles de la forme (_,(Nec _))*)

let necn l = rev( fold_left (fun l b -> match b with | (n,(Nec _)) -> b ::l | _ -> l) [] l);;

let valid a =
  (* a est une formule intuitioniste 
     a est traduite en une formule as4 de s4
     si as4 est valide, donne une preuve (justifiée) de cette validité
     sinon donne un contre-modèle de a *)
  let as4 = s4_of_int a in
  let proof = {case = []; label = Assumption (1,Not as4); numero = 1; from = 0; from2 = 0; 
	       successor = []} in
    (* proof est l'arbre de preuve *)
  let lbo = ref [{domain = 1; last_number = 1; 
		  relation = []; leaf = proof; todo = [1,(1,Not as4)];is_done = []}] in
    (* lbo est la liste des branches ouvertes *)
    
    while (!lbo <> []) &&   (hd !lbo).todo <>[] do
      let left_branch = hd !lbo in
      let lan = left_branch.todo @ left_branch.is_done in
      let la = put_off_first_component (lan) in
      let (b,(e,(p,q))) = contradiction lan in
	if b then 
	  ( (* on a ajoute l'hypothèse (e,False) à la preuve *)
	    let current_leaf = left_branch.leaf and k = left_branch.last_number in
	    let p1 = {case = current_leaf.case; numero = k+1;
		      from = min p q; from2 = max p q;
		      label = Assumption (e,False);successor = []} in
	      current_leaf.successor <- [p1];
	    (* on enlève la branche fermée *)
	    lbo := tl(!lbo)
	  )
	else if falseonbranch lan  then 
	    ( (* on enlève la branche fermée *)
	      lbo := tl(!lbo)
	    )
	else
	  (* appliquons une règle *)
	
	  let 
	      k = left_branch.last_number and 
	      (* tri des hypothèses avant l'application des règles en première hypothèse ph et
		 autres hypothèses ah : avec les priorités choisies la règle du Possible n'est
		 applicables que si aucune autre règle n'est applicable *)
	      ((n,(p,f)) as ph):: ah = List.sort infeg_assumption left_branch.todo 
	  in
	    match f with

		(* litterals *)

	      | (Var _)  ->
		  left_branch.todo <- ah ; left_branch.is_done <- ph::left_branch.is_done ;

	      | (Not (Var _))  ->
		  left_branch.todo <- ah ; left_branch.is_done <- ph::left_branch.is_done ;

		  (* elimination de (Not False *)
	      | (Not False) ->
		  left_branch.todo <- ah ; left_branch.is_done <- left_branch.is_done ;
		
	      (* formules conjonctive et déplacement négation *)

	      | (And (a,b)) ->
		  left_branch.todo <-  (k+1,(p, a))::((k+2,(p, b)):: ah) ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+2;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1; from = n; from2=0;
			    label = Assumption (p,a); successor = []} 
		  and p2 = {case = current_leaf.case; numero = k+2; from = n; from2=0;
			    label = Assumption (p,b); successor = []} 
		  in
		    current_leaf.successor <- [p1] ; p1.successor <- [p2];
		    left_branch.leaf <- p2;

	      | (Not (Or (a, b))) ->
		  left_branch.todo <-  (k+1,(p, Not a))::((k+2,(p, Not b)):: ah) ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+2;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1; from = n; from2 = 0;
			    label = Assumption (p,Not a); successor = []} 
		  and p2 = {case = current_leaf.case; numero = k+2; from = n; from2 = 0;
			    label = Assumption (p,Not b); successor = []} 
		  in
		    current_leaf.successor <- [p1] ; p1.successor <- [p2];
		    left_branch.leaf <- p2;

	      | (Not (Imp (a, b))) ->
		  left_branch.todo <-  (k+1,(p, a))::((k+2,(p, Not b)):: ah) ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+2;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1; from = n; from2 = 0;
			    label = Assumption (p,a); successor = []} 
		  and p2 = {case = current_leaf.case; numero = k+2; from = n; from2 = 0;
			  label = Assumption (p,Not b); successor = []} 
		  in
		    current_leaf.successor <- [p1] ; p1.successor <- [p2];
		    left_branch.leaf <- p2;
		 
	      | (Not (Not a)) ->
		  left_branch.todo <-  (k+1,(p, a)):: ah ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+1;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1 ; from = n;from2 = 0;
			  label = Assumption (p,a); successor = []}  
		  in
		    current_leaf.successor <- [p1] ; 
		    left_branch.leaf <- p1;

	      | (Not (Nec a)) ->
		  left_branch.todo <-  (k+1,(p, Pos (Not a))):: ah ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+1;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1 ; from = n;from2 = 0;
			  label = Assumption (p,Pos (Not a)); successor = []}  
		  in
		    current_leaf.successor <- [p1] ; 
		    left_branch.leaf <- p1;

	      | (Not (Pos a)) ->
		  left_branch.todo <-  (k+1,(p, Nec (Not a))):: ah ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+1;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1 ; from = n;from2 = 0;
			    label = Assumption (p,Nec (Not a)); successor = []}  
		  in
		    current_leaf.successor <- [p1] ; 
		    left_branch.leaf <- p1;



	      | (Equ (a,b)) -> 
		  left_branch.todo <-  (k+1,(p, Imp(a,b)))::((k+2,(p, Imp(b,a))):: ah) ;
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+2;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1; from = n; from2 = 0;
			  label = Assumption (p,Imp(a,b)); successor = []} 
		  and p2 = {case = current_leaf.case; numero = k+2; from = n; from2 = 0;
			  label = Assumption (p,Imp(a,b)); successor = []} 
		  in
		    current_leaf.successor <- [p1] ; p1.successor <- [p2];
		    left_branch.leaf <- p2;

	      | (Nec a) ->
		  (* reflexivité de Nec *)
		  left_branch.todo <-  (k+1,(p, a)):: ah ; 
		  left_branch.is_done <- ph :: left_branch.is_done;
		  left_branch.last_number <- k+1;
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in 
		  let p1 = {case = current_leaf.case; numero = k+1 ; from = n; from2 = 0;
			  label = Assumption (p,a); successor = []}  
		  in
		    current_leaf.successor <- [p1] ; 
		    left_branch.leaf <- p1;	      
		       

	      (* formules disjonctives *)
	      | (Or (a,b)) ->
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in
		  let p1 = {case = current_leaf.case@[1]; numero = k+1; from = n; from2 = 0;
			    label = Assumption (p,a); successor = []}
		  and p2 = {case = current_leaf.case@[2]; numero = k+1; from = n; from2 = 0;
			    label = Assumption (p,b); successor = []} in
		    current_leaf.successor <- [p1;p2] ;
		    let d = left_branch.domain and r = left_branch.relation 
					       and current_is_done = left_branch.is_done in 
		    let first_branch = 
		      {domain = d; last_number = k+1; relation = r; leaf = p1; 
		       todo = (k+1,(p,a))::ah; is_done = ph::current_is_done}
		    and second_branch =
		      {domain = d; last_number = k+1; relation = r; leaf = p2; 
		       todo = (k+1,(p,b))::ah; is_done = ph::current_is_done}
		    in lbo := first_branch::second_branch::(tl !lbo);

	      | (Imp (a,b)) ->
		
		  (* Modification de la preuve *)
		  let current_leaf = left_branch.leaf in
		let p1 = {case = current_leaf.case@[1]; numero = k+1; from = n;from2=0;
			  label = Assumption (p,Not a); successor = []}
		and p2 = {case = current_leaf.case@[2]; numero = k+1; from = n;from2=0;
			  label = Assumption (p,b); successor = []} in
		  current_leaf.successor <- [p1;p2] ;
		  let d = left_branch.domain and r = left_branch.relation 
					     and current_is_done = left_branch.is_done in 
		  let first_branch = 
		    {domain = d; last_number = k+1; relation = r; leaf = p1; 
		     todo = (k+1,(p,Not a))::ah; is_done = ph::current_is_done}
		  and second_branch =
		    {domain = d; last_number = k+1; relation = r; leaf = p2; 
		     todo = (k+1,(p,b))::ah; is_done = ph::current_is_done}
		  in lbo := first_branch::second_branch::(tl !lbo);

	    | (Not (And (a,b))) ->
		(* Modification de la preuve *)
		let current_leaf = left_branch.leaf in
		let p1 = {case = current_leaf.case@[1]; numero = k+1; from = n;from2 = 0;
			  label = Assumption (p,Not a); successor = []}
		and p2 = {case = current_leaf.case@[2]; numero = k+1; from = n;from2 = 0;
			  label = Assumption (p,Not b); successor = []} in
		  current_leaf.successor <- [p1;p2] ;
		  let d = left_branch.domain and r = left_branch.relation 
					     and current_is_done = left_branch.is_done in 
		  let first_branch = 
		    {domain = d; last_number = k+1; relation = r; leaf = p1; 
		     todo = (k+1,(p,Not a))::ah; is_done = ph::current_is_done}
		  and second_branch =
		    {domain = d; last_number = k+1; relation = r; leaf = p2; 
		     todo = (k+1,(p,Not b))::ah; is_done = ph::current_is_done}
		  in lbo := first_branch::second_branch::(tl !lbo);

	    | (Not (Equ(a,b))) ->
		(* Modification de la preuve *)
		let current_leaf = left_branch.leaf in
		let p1 = {case = current_leaf.case@[1]; numero = k+1; from = n;from2 = 0;
			  label = Assumption (p,Not (Imp(a,b))); successor = []}
		and p2 = {case = current_leaf.case@[2]; numero = k+1; from = n;from2 = 0;
			  label = Assumption (p,Not (Imp(b,a))); successor = []} in
		  current_leaf.successor <- [p1;p2] ;
		  let d = left_branch.domain and r = left_branch.relation 
					     and current_is_done = left_branch.is_done in 
		  let first_branch = 
		    {domain = d; last_number = k+1; relation = r; leaf = p1; 
		     todo = (k+1,(p,Not (Imp(a,b))))::ah; is_done = ph::current_is_done}
		  and second_branch =
		    {domain = d; last_number = k+1; relation = r; leaf = p2; 
		     todo = (k+1,(p,Not (Imp(b,a))))::ah; is_done = ph::current_is_done}
		  in lbo := first_branch::second_branch::(tl !lbo);

	    | Pos a ->
		(* Rappel : 
		   (p,f) est la première hypothèse à traiter
		   ah est le reste des  hypothèses numérotés à traiter 
		   la est la liste de hypothèses sans les numéros
		*)
		let d = left_branch.domain and r = left_branch.relation in
		let tm = rearrange d la in 
		let tmn = rearrangen d lan in
		  (* pour i de 1 à d, tmn(i) est la liste des formules numérotées de la
		     branche dans l'état i *)
		let lnecn = necn (tmn.(p)) in
		  (* lnecn est la liste des formules numérotées (Nec _) de l'état p *)
		let lnec = put_off_first_component lnecn in		  
		  try 
		    (existe (a::lnec) tm);
		    (* l'exception n'ayant pas été levée,
			 il n'y a pas d'états contenant les formules de la 
			 liste a::lnec *)
		    (* on ajoute un nouvel état d+1 où toutes
			 les formules de a::lnec sont supposées vraies 
		    *)
		    left_branch.domain <- d+1; left_branch.relation <- (p,d+1)::r;
		    left_branch.is_done <- ph::left_branch.is_done;
		    (* ajout de la nouvelle flêche *)
		    (let current_leaf = left_branch.leaf in
		     let p1 = {case = current_leaf.case; label = Arrow (p,d+1);
			       numero = k+1; from = n; from2 = 0; successor = []} in
		       current_leaf.successor <- [p1];
		       left_branch.leaf <- p1;);
		    (* ajout de la nouvelle formule *)
		    (let current_leaf = left_branch.leaf in
		     let p1 = {case = current_leaf.case; label = Assumption (d+1,a);
			       numero = k+2; from = n; from2 = 0; successor = []} in
		       current_leaf.successor <- [p1];
		       left_branch.leaf <- p1;
		       left_branch.todo <- (k+2,(d+1,a))::left_branch.todo;
		    );
		      
		    (* pour b la j-eme formule de la liste lnecn,
		       on ajoute sur la branche l'hypothèse (d+1,b) de numéro k+1+j *)
		    let num = ref (k+3) and l = ref lnecn in
		      while !l <> [] do	  
			(let (n,b) = hd (!l) in
			   left_branch.todo <- (!num,(d+1,b))::left_branch.todo;
			   let current_leaf = left_branch.leaf in
			   let p1 = {case=current_leaf.case; label =  Assumption (d+1,b);
				     numero = !num ; from = k+1; from2 = n ;successor = []} in
			     current_leaf.successor <- [p1];
			     left_branch.leaf <- p1;);
			l := tl !l;num := !num+1;
		      done;
		      left_branch.last_number <- !num -1

		  with Index i ->
		    (* ajout (s'il n'existe pas) de l'arc (p,i) 
		       à la branche gauche et à l'arbre *)
		    left_branch.todo <- ah; left_branch.is_done <- ph::left_branch.is_done;
		    if (p <> i) && not ((mem (p,i) r)) then
		      (
			(* modification de la branche gauche *)
			left_branch.relation <- (p,i)::r;
			left_branch.last_number <- k+1;
			(* modification de l'arbre de la preuve :
			   totalement inutile : cet arc ne sert à rien 
			   dans la preuve de validité 


						
			let current_leaf = left_branch.leaf in
			let p1 = {case = current_leaf.case; label = Arrow (p,i);
			   numero = k+1; from = n; from2 = 0; successor = []} in
			   current_leaf.successor <- [p1];
			  left_branch.leaf <- p1
			*)
		      )

    done;
    if !lbo = [] 
    then
      ( print_string "The formula "; print_newline ();
	Print.print_formula a; print_newline ();
	print_string "is valid"; print_newline ();
	print_string "Proof that the negation of its S4-translation is unsatisfiable";
	print_newline ();
	Print.print_proof proof ;)
    else 
      ( print_string "The formula "; print_newline ();
	Print.print_formula a;print_newline ();
	print_string "is not valid"; print_newline ();
	print_string "Formula's counter-model "; print_newline ();
	Print.print_assign (hd !lbo))
;;
		  
		  

