
(* ===========================================================================================
=======================  RESOLUTION     ======================================================
============================================================================================*)

open Types;;
open Listc;;
	       
let oppose l1 l2 = 
  (* résultat :  true si et seulement si l1 et l2 sont des littéraux opposés *)
  match (l1, l2) with
    | (Pos p1, Neg p2) -> p1 = p2
    | (Neg p1, Pos p2) -> p1 = p2
    | _ -> false ;;

let opp = 
  (* résultat : le littéral opposé à l *)
  function 
    | (Pos x) -> Neg x
    | (Neg x) -> Pos x
;;

let var l = 
  (* résultat :  si l est un littéral positif alors l sinon l'opposé de l *)
    match l with
    | (Pos p1) as l1 -> l1
    | (Neg p1) -> Pos p1 ;;


let rec valide e = 
  (* résultat : true ssi e est une liste valide de littéraux *)
  match e with
   | [] -> false 
   | t::f -> (List.mem (opp t) f) || valide f;;

let valide_clause c = 
  (*résultat : true ssi c est une clause valide  *)
  valide c.corps ;;

let suppression_valides  l = 
  (* enlève de l les clauses  valides *)
  suppression valide_clause l;;



(* =======================================================================================
==================================== CALCUL DU RESOLVANT (1° ordre)  =====================
======================================================================================= *)


(* ================================== SUBSTITUTION (1° ordre) ========================= *)

(* Une substitution est une liste de paires string * terme 
   Une équation est un couple de termes ou de littéraux *)

let subst_terme x t e =
  (* remplace (Var x) par le terme t dans le terme e *)
  let rec subst = function
    |((Var y) as u) -> if x = y then t else u
    | (Comp (y,ly)) -> Comp (y, List.map subst ly)
  in subst e;;

let subst_eqs x t le = 
  (* remplace (Var x) par t dans toutes les équations entre termes de la liste le *) 
  List.map (fun (u,v) ->(subst_terme x t u),(subst_terme x t v)) le;;

let subst_res x t lr = 
  (* remplace (Var x) par t dans les termes de la substitution lr *)
  List.map (fun (u,v) -> u, (subst_terme x t v)) lr ;;

let subst_t s t =
  (* applique la substitution s au terme t *)
  let rec subst_t_aux = function
    |((Var x) as y) -> (try (List.assoc x s) with Not_found -> y)
    | (Comp(x,lx)) -> Comp(x, List.map subst_t_aux lx)
  in subst_t_aux t;;
	
let subst_l s x =
  (* applique la substitution s au littéral x *)
  match x with
    | (Pos x) -> (Pos (subst_t s x))
    | (Neg x) -> (Neg (subst_t s x)) ;;

(* ================================= FIN SUBSTITUTION ==================================== *)
			


(* ================================== UNIFICATION (1° ordre) ============================= *)

exception Non_unifiable;;

let occur v t = 
  (* lève l'exception Non_unifiable si v est une variable du terme t *)
  let rec aux = function
    | (Var v') -> if v = v' then raise Non_unifiable
    | (Comp (_,la)) -> List.iter aux la
  in aux t;;	

let rec trans e  s  = 
  (* calcule par une méthode assez lente l'instance principale s' de la substitution s 
     où s' est solution du système e d'équations entre termes 
     si s' n'existe pas, l'exception Non_unifiable est levée *)
match e with
|[] -> s
| (t1,t2)::le ->
    if t1 = t2 then trans le s
    else
      match (t1,t2) with
	| (Var x,  t) -> occur x t; 
	    trans (subst_eqs x t le) (let sp =(subst_res x t s) in if List.mem (x,t) sp then sp else (x,t)::sp) 
	| (t, Var x) -> occur x t; 
	    trans (subst_eqs x t le) (let sp =(subst_res x t s) in if List.mem (x,t) sp then sp else (x,t)::sp)
	|  (Comp(x,lx),Comp(y,ly)) -> 
	     if x = y && (List.length lx) = (List.length ly)
	     then (trans ((List.combine lx ly) @ le)  s) 
	     else raise Non_unifiable
;;	
	

let unift t1 t2 = 
  (* si les termes t1 et t2 sont unifiables, cette fonction en
     calcule l'unificateur principal, sinon elle lève l'exception Non_unifiable *)
  trans [t1,t2] [];;	

let rec equations_de_liste l =
  (* transforme la liste [x1; x2; x3; ...] en une liste de couples [x1,x2; x2,x3; ...]*)
  match l with
    | [] -> []
    | [_] -> []
    | x1::x2::f -> (x1,x2)::equations_de_liste (x2::f);;

let equation_de_termes_de_equations_de_litterals e =
  (* change une équation entre littérals en une équation entre termes
     lève l'exception Non_unifiable s'il n'y a pas de solution *)
  match e with
    | Pos t1, Pos t2 -> t1,t2
    | Neg t1, Neg t2 -> t1,t2
    | _ -> raise Non_unifiable;;

let unifl l =
  (* si la liste l de littéraux est unifiable, en calcule l'unificateur principal
     sinon lève l'exception Non_unifiable *)
  let l1 = equations_de_liste l in
  let l2 = List.map equation_de_termes_de_equations_de_litterals l1 in
    trans l2 [];;

(* ============================== FIN UNIFICATION ========================================== *)



(* ============================ FACTEURS D'UNE CLAUSE (1° ordre) =========================== *)

let facteurs c =
  (* résultat : la liste des facteurs de c, c compris *)
  let l1 = sous_ens2 c.corps in
    (* l1 est la liste des listes qui sont des sous-listes de la liste des littéraux de c 
       et qui ont au moins  2 éléments *)
    c :: List.fold_left 
      (fun a b -> 
	 try let s = unifl b in 
	   {numero = 0; origine = Facteur (c,s); 
	    corps = ens_de_liste (List.map (subst_l s) c.corps)}::a
	 with Non_unifiable -> a) [] l1;;
	     
(* ================================ FIN FACTEURS =========================================== *)
    	


(* ============================ RENOMMAGE D'UNE CLAUSE (1° ordre) ========================== *)

let nv l =
  (* produit une chaîne "v{chiffre}" non élément de la liste l *)
  let rec aux i = let s = "v"^(string_of_int i) in 
    if (List.mem s l) then aux (i+1) else s
  in aux 0;;

let rec renommage e f = 
  (* e est une liste de chaînes sans répétition, 
     la fonction calcule un renommage de domaine e par des variables
     qui ne figurent pas dans f *)
  match e with
    | [] -> []
    | t::fin -> 
	if List.mem t f 
	then let x = nv f in (t, Var x ):: (renommage fin (x::f))
	else renommage  fin f ;;

let rec vars_terme t =
  (* liste des variables du terme t *)
  match t with
    | Var x -> [x]
    | Comp(_,l) -> List.fold_left (fun a b -> a @ (vars_terme b)   ) [] l;;

let vars_litteral l =
  (* liste des variables du littéral l *)
  match l with
    | Pos t -> vars_terme t
    | Neg t -> vars_terme t;;

let vars_litteraux c =
  (* liste sans répétition des variables de la liste c de litteraux *)
  ens_de_liste (List.fold_left (fun a b -> a @ (vars_litteral b)   ) [] c);;



(* ============================== FIN RENOMMAGE ===================================== *)



(* ============================ RESOLVANT BINAIRE (1° ordre) ======================== *)

let res_aux c1 l1 c2 l2 =
  (* l1 est un littéral de c1, l2 est un littéral de c2,
     résultat : 
     - si l1 et (opp l2) ont s comme unificateur principal , alors [r]
     où r est le résolvant de corps  s((c1-{l1})U(c2-{l2}))
     - si l1 et (opp l2) ne sont pas unifiables, alors []
  *)
  try let s = unifl [l1; opp l2] in
  let cc = (enlever l1 c1.corps)@(enlever l2 c2.corps) in
  let cr = List.map (subst_l s) cc in
  [{numero = 0; origine = ResBin(c1,c2,s); corps = ens_de_liste cr}]
    (* avec ens_de_liste plusieurs littéraux identiques peuvent être remplacés par un seul *)
  with Non_unifiable -> [];;

let res_bin c1 c2 =
  (* résultat : liste des résolvants binaires des clauses c1 et c2  *)
  List.fold_left
    (fun lr l1 ->  
       lr @ 
       (List.fold_left 
	  (fun lr l2 -> lr @ (res_aux c1 l1 c2 l2)) [] c2.corps)) [] c1.corps;;






let resolvant_sans_copie c1 c2 =
  (* pré : les clauses c1 et c2 n'ont pas de variables communes
     résultat : la liste de tous les résolvants de c1 et c2 *)
  (* on construit tous les facteurs de c1 et de c2 et on calcule les résolvants binaires
     entre ces facteurs *)
  let lf1 = facteurs c1 and lf2 = facteurs c2 in
    List.fold_left
      (fun lr c ->  
	 lr @ 
	 (List.fold_left 
	    (fun lr d -> lr @ (res_bin c d)) [] lf2)) [] lf1;;





let copie c r = 
  (* construit une copie de la clause c avec le renommage r *)
  {numero = 0; origine = Copie (c,r); corps = List.map (subst_l r) c.corps}
	
    


(* définition de restrictions de la résolution : au moins un des parents doit être une
clause positive, au moins un des parents doit être une clause négative *)

let type_resolution = ref 0;;
 
let pos = 1;;

let neg = -1;;

let nor = 0;;

let positive c =
  (* résultat : true si  c est une clause positive *)
  List.for_all (function | Pos _ -> true | _ -> false) c.corps ;;

let negative c = 
  (* résultat : true si  c est une clause négative *)
  List.for_all (function | Neg _ -> true | _ -> false) c.corps ;;

let resolution_autorisee t c1 c2 =
  (* résultat : true si la résolution de type t est autorisée entre c1 et c2 *)
  t = neg && ((negative c1) || (negative c2))
    || t = pos && ((positive c1) || (positive c2))
    || t = nor;;

(* calcul des résolvants : restait de 2005 à 2019 une erreur dans l'appel de renommage *)

let resolvants c1 c2 =
  (* résultat : la liste des résolvants des clauses  c1 et c2 *)
  if resolution_autorisee !type_resolution c1 c2
  then
    begin
      (* copie de c2 si necessaire *)
      let vc1 = vars_litteraux c1.corps
      and vc2 = vars_litteraux c2.corps in
        (* Avant il y avait l'erreur (renommage vc2 vc1) *) 
	if disjoint vc1 vc2 then resolvant_sans_copie c1 c2
	else 
	  let c2copie = copie c2 (renommage vc2 (vc1@vc2)) in
	    resolvant_sans_copie c1 c2copie
    end
  else [];;


(* restriction incomplète de la résolution : contrôle de la longueur des clauses et de la hauteur des clauses
on supprime toutes les clauses dépassant ces bornes *)



let rec hauteur_terme t = 
  (* résultat : hauteur d'un terme t *)
  match t with 
    | Var _ -> 0
    | Comp (_,[]) -> 0
    | Comp (_,lt) -> 1+ (List.fold_left (fun m1 t -> max (hauteur_terme t) m1) 0 lt);;

let hauteur_litteral l =
  (* resultat : hauteur d'un littéral *)
  match l with
    | Pos t -> hauteur_terme t
    | Neg t -> 1 + hauteur_terme t;;

let hauteur_clause c =
  (* résultat : hauteur d'une clause c *)
  List.fold_left (fun m l -> max (hauteur_litteral l) m) 0 c.corps;;
  




(* ===================================== FIN RESOLVANT =============================== *)
